package server.censor.signature.experiment;

import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;

import communication.CqeType;

import notification.serverToClient.QueryResultNotification;
import exception.AutomatonException;
import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.LoadCensorException;
import exception.ParserException;
import exception.ProverException;
import exception.UnexpectedNotificationException;
import exception.UnknownUserException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;
import server.censor.signature.StaticSignatureCensor;
import server.censor.signature.util.MaintenanceUtil;
import server.core.Client;
import server.core.Server;
import server.data.User;
import server.database.AppDatabase;
import server.database.DatabaseConfiguration;
import server.database.NewMaintenanceDatabase;
import server.database.OracleSQLAppDatabase;
import server.database.schema.Schema;
import server.util.Tuple;
import server.parser.Formula;
import server.parser.MumFormula;

/**
 *  Diese Klasse dient der Ausfuehrung des Experiments zum SignatureCensor.
 *  Das Experiment ist in "Signature-Based Inference-Usability Confinement for Relational 
 *  Databases under Functional and Join Dependencies" von Biskup etc. beschrieben.
 *  Details des Experimentaufbaus koennen der Datei 'Experimentbeschreibung_DBSEC12.txt' 
 *  entnommen werden.
 */
public abstract class ExperimentExecution {

	/** Logger */
	private final static Logger logger = Logger.getLogger("edu.udo.cs.ls6.cie.server.censor.signature");
	
	/** Server */
	protected Server server;
	
	/** Client */
	protected Client client;
	
	/** Maintenance Datenbank */
	protected NewMaintenanceDatabase maintenanceDB;
	
	/** Applikations Datenbank */
	protected AppDatabase appDB;
	
	/** Hilfsklasse fuer Maintenance-Aufgaben */
	protected MaintenanceUtil maintenanceUtil;
	
	/** Verwendeter Zensor */
	protected StaticSignatureCensor signatureCensor;
	
	/** Verwendung des flexiblen oder unflexibler Ansatzes */
	protected boolean flexible;
	
	/** UserID des Users fuer den das Experiment durchgefuehrt wird */
	protected int userID;
	
	/** Detaillierte oder kummulierte Statistiken */
	protected boolean cumulatedStatistics;
	
	/** Name der zu schuetzenden Relation */
	protected String relationname;
	
	/**
	 * Konstruktor. Stellt die Verbindung zur App- und Maintenance-Datenbank her und erstellt eine Instanz des Automaten, die fuer die Durchfuehrung
	 * des Experiments verwendet wird.
	 * 
	 * @param dbConfig Datenbank-Konfiguration
	 * @param flexible true, wenn der flexible Ansatz verwendet werden soll, false sonst
	 * @param userIdUnflexible Id des Users fuer den unflexiblen Ansatz
	 * @param usernameUnflexible Benutzername zur Anmeldung an der Datenbank fuer den unflexiblen Ansatz
	 * @param passwordUnflexible Passwort zur Anmeldung an der Datenbank fuer den unflexiblen Ansatz
	 * @param userIdFlexible Id des Users fuer den flexiblen Ansatz
	 * @param usernameFlexible Benutzername zur Anmeldung an der Datenbank fuer den flexiblen Ansatz
	 * @param passwordFlexible Passwort zur Anmeldung an der Datenbank fuer den flexiblen Ansatz
	 */
	public ExperimentExecution(Tuple<DatabaseConfiguration,DatabaseConfiguration> dbConfig, String relation, boolean flexible, int userID, boolean cumulatedStatistics) throws DatabaseException, IOException {
		NewMaintenanceDatabase mDB = NewMaintenanceDatabase.load( dbConfig.getFirstElement() );
		AppDatabase appDB = new OracleSQLAppDatabase( dbConfig.getSecondElement() );
			
		this.init(mDB,  appDB, relation, flexible, userID, cumulatedStatistics);
	}
	
	public ExperimentExecution(NewMaintenanceDatabase maintenanceDB, AppDatabase appDB, String relation, boolean flexible, int userID, boolean cumulatedStatistics) throws DatabaseException, IOException {
		this.init(maintenanceDB,  appDB, relation, flexible, userID, cumulatedStatistics);
	}
	
	private void init(NewMaintenanceDatabase maintenanceDB, AppDatabase appDB, String relation, boolean flexible, int userID, boolean cumulatedStatistics) throws DatabaseException, IOException {
		this.relationname = relation;
		this.userID = userID;
		this.flexible = flexible;
		this.cumulatedStatistics = cumulatedStatistics;
		
		this.maintenanceDB = maintenanceDB;
		this.appDB = appDB;
		
		// Initialize server dummy (don't start it, just allocate memory).
		server = new Server(4321);
		
		client = new Client(server, null);
		client.setMaintenanceDB( this.maintenanceDB );
		client.setApplicationDB( this.appDB );
		
		// Load the user.
		try {
			User user = client.getMaintenanceDB().getUserManagement().load( this.userID );
			client.setUser( user );
		} catch( UnknownUserException e ) {
			throw new DatabaseException("Unknown user! User with id " + this.userID + " doesn't exist.", null);
		}
		
		this.maintenanceUtil = new MaintenanceUtil();
		
		if( this.flexible )
			this.signatureCensor = (StaticSignatureCensor)server.getCensorManagement().getCensor("FlexibleStaticSignature");
		else
			this.signatureCensor = (StaticSignatureCensor)server.getCensorManagement().getCensor("UnflexibleStaticSignature");
	}
	
	/**
	 * Stoesst die Auswertung der uebergebenen Interaktion an und liefert das zugehoerige Ergebnis
	 * zurueck.
	 * 
	 * @param interaction	Interaktion in Form einer Formel. Dies ist der Query, der vom Signatur-Zensor ausgewertet werden soll.
	 * @return Ergebnis der Auswertung
	 * @throws DatabaseException
	 * @throws LoadCensorException
	 * @throws UnexpectedNotificationException
	 * @throws AutomatonException
	 * @throws ProverException
	 * @throws ParserException
	 * @throws UnsupportedFormulaException
	 * @throws DictionaryTooSmallException
	 * @throws UnsupportedConfidentialityPolicyException
	 * @throws IOException
	 */
	private QueryResultNotification submitQuery(Formula interaction) throws Exception {
		// FIXME: Auswertung ueber den neuen Automaten durchfuehren, wenn dieser fertig implementiert ist
		
		// Zeitmessung starten
		long startTime = System.currentTimeMillis();
		
		QueryResultNotification result = signatureCensor.evaluate(client, interaction);
		
		// Zeitmessung abschliessen
		long endTime = System.currentTimeMillis();
		result.setProcessingTime(endTime-startTime);
		
		return result;
	}
	
	protected abstract List<Formula> randomQuerys(int numberOfQueries) throws DatabaseException, ParserException;
	
	
	/**
	 * Fuehrt das Experiment durch. Dies setzt voraus, dass das Experiment bereits durch Aufruf der Methode
	 * prepareExperiment vorbereitet worden ist.
	 * Die Auswertung des Experiments (Statistiken) werden in eine Tabelle in der Datenbank geschrieben.
	 * 
	 * @param runID				verwendete ID unter der der Durchlauf in der Auswertungstabelle gespeichert wird
	 * @param numberOfQuerys	Anzahl Querys, die im Durchlauf an die Datenbank gestellt werden
	 * @throws IOException 
	 * @throws UnsupportedConfidentialityPolicyException 
	 * @throws DictionaryTooSmallException 
	 * @throws UnsupportedFormulaException 
	 * @throws ProverException 
	 * @throws AutomatonException 
	 * @throws UnexpectedNotificationException 
	 * @throws LoadCensorException 
	 */
	public void executeExperiment(int runID, int numberOfQuerys) throws DatabaseException, ParserException, LoadCensorException, UnexpectedNotificationException, AutomatonException, ProverException, UnsupportedFormulaException, DictionaryTooSmallException, UnsupportedConfidentialityPolicyException, IOException, Exception {
		// clear log
		// FIXME, nur zum Speicherplatz sparen (Datenbank-Speicherplatz war/ist leider bzgl. groesserer Instanzen stark limitiert), ansonsten nicht notwendig
		client.getUser().getLog().clear();
		
		// Flags zuruecksetzen
		if( flexible )
			this.maintenanceUtil.clearFlagsFlexible(this.maintenanceDB.getDb(), relationname, this.userID);
		else
			this.maintenanceUtil.clearFlagsUnflexible(this.maintenanceDB.getDb(), relationname, this.userID);
		
		// zufaellige Anfragen generieren
		List<Formula> querys = this.randomQuerys(numberOfQuerys);
		
		// zufaellige Anfragen auswerten
		int queryID = 1;
		for( Formula interaction : querys ) {
			logger.info("Submitting query " + queryID + ".");
			
			interaction.setInteractionType(CqeType.InteractionType.QUERY);
			QueryResultNotification result = this.submitQuery(interaction);
			
			String resultFormula = "";
			if( result.getResult().isEmpty() )
				resultFormula = "fail - no result";
			else {
				Formula f = result.getResult().get(0).getFormula();
				if( f instanceof MumFormula )
					resultFormula = "mum";
				else if( f.equals(interaction) )
					resultFormula = "true";
				else
					resultFormula = "false";
			}
			String databaseProcessingTime;
			if( flexible )
				databaseProcessingTime = this.maintenanceUtil.getDatabaseProcessingTimeFlexible(this.maintenanceDB.getDb());
			else
				databaseProcessingTime = this.maintenanceUtil.getDatabaseProcessingTimeUnflexible(this.maintenanceDB.getDb(), relationname);
			
			if( cumulatedStatistics )
				this.maintenanceUtil.experimentEvaluation_writeQueryExecutionStatisticsAccumulated(this.maintenanceDB.getDb(), runID, queryID, interaction.toString(), resultFormula, result.getProcessingTime(), Integer.parseInt(databaseProcessingTime));
			else
				this.maintenanceUtil.experimentEvaluation_writeQueryExecutionStatistics(this.maintenanceDB.getDb(), runID, queryID, interaction.toString(), resultFormula, result.getProcessingTime(), Integer.parseInt(databaseProcessingTime));
			
			++queryID;
			
			if( flexible && queryID % 1000 == 0 ) {
				// alle 1000 Anfragen die Anfrage-Tabelle loeschen (nur zum Speicherplatz sparen)
				this.maintenanceDB.getDb().truncateTable( Schema.maintenanceDatabase.signaturesFlexibleQuery.name );
				this.maintenanceDB.getDb().truncateTable( Schema.maintenanceDatabase.signaturesFlexibleQueryAV.name );
			}
		}
	}
}
