package server.censor.signature;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import exception.DatabaseException;
import exception.UnsupportedFormulaException;
import server.censor.QueryCensor;
import server.censor.signature.util.MaintenanceUtil;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.MumFormula;
import server.parser.node.ConstantNode;
import server.parser.node.Node;
import server.parser.node.RelationnameNode;

/**
 * Oberklasse für den Zensor, der die statische signaturbasierte Zugriffskontrolle realisiert.
 * (s. Diplomarbeit: Torsten Schlotmann / Kontrollierte Auswertung relationaler Anfragen mit Hilfe von aus Integritätsbedingungen
 * erzeugten Signaturen)
 */

public abstract class StaticSignatureCensor extends QueryCensor {

	protected MaintenanceUtil maintenanceUtil;
	
	public StaticSignatureCensor(String censorName, Server server) throws DatabaseException {
		super(censorName, server);
		
		// FIXME: muesste obsolete sein, weil MaintenanceDB jetzt immer relational ist (Dirk)
		// Sicherstellen, dass die Datenbank eine RelationalMaintenanceDatabase ist
		//if( ! this.getMaintenanceDB().isRelationalMaintenanceDatabase() )
		//	throw new DatabaseException("Unsupported Database! StaticSignatureCensor requires a relational maintenance database.", null);
		
		this.maintenanceUtil = new MaintenanceUtil();
	}
	
	@Override
	protected List<Formula> run( Client client, Formula interaction ) throws DatabaseException, UnsupportedFormulaException {
		if( ! interaction.getLanguageTypes().contains( "existential_r_sentence" ) ) {
			// Ungueltige Anfrage
			throw new UnsupportedFormulaException("Unsupported formula! The SignatureCensor can only handle existential-R-sentences: " + interaction.toString() + " (language types of the formula: " + interaction.getLanguageTypes() + ")");
		}
		
		Vector<Formula> answer = new Vector<Formula>();
		// Array zum aufsammeln der einzelnen Teile der Anfrage
		ArrayList<String> CQE_Query = new ArrayList<String>();
		// Baum zur uebergebenen Anfrage (Knoten nach RootNode als Wurzel)
		Node node = interaction.getRootChild();
		
		// aufgrund der eingeschraenkten Sprache nur geschlossene Anfragen moeglich
		// demnach kann die Wurzel "maximal" einen existenzquantifizierten Knoten repraesentieren
		// falls dies so ist, wird das Kind geholt, welches dann einem relationalen Knoten entspricht
		if ( node.isExistentialQuantifiedNode() ) {
			node = node.getChild(0);
		}
		
		for (Node child : node.getChildren() ) {
			// alle Variablen werden im PL/SQL-Code ueber den String 'X' identifiziert
			// Variablen in existenzquantifizierten Formeln in der Datenbank (statisch: Relation SignaturenStat, dynamisch: Relation SignaturenGen)
			// entsprechend bezeichnen
			if ( child.isVariableNode() ) {
				CQE_Query.add( "*" );
			}
			// alle anderen Kinder entsprechen aufgrund der eingeschraenkten Anfragesprache Konstanten
			if ( child.isConstantNode() ) {
				child = (ConstantNode) child;
				CQE_Query.addAll( child.getConstants());
			}
		}

		// Relationenname anhang gestellter Anfrage auslesen
		String relation = ((RelationnameNode) node.getChild(0)).getRelationname();	
		
		// Spaltennamen entsprechend ausgelesener Relation
		String[] ColumnNames = client.getApplicationDB().getAttributeNames(relation.toString());
		
		// INSERT zusammenbasteln und an Datenbank stellen -> Trigger und entsprechend dahinterliegender PL/SQL-Code werden
		// ausgefuehrt
		
		// es stehen 2 Moegliche Signatur-Zensoren in Form von Triggern in der Datenbank zur Verfuegung: flexible und unflexible Version
		// welche dieser beiden Versionen ausgewaehlt wird haengt von der konkreten Unterklasse ab
		String result = this.writeToCQEQuery( client, CQE_Query, interaction, ColumnNames, relation );
		
		if( result.equals("mum") ) {
			// verweigere Antwort
			answer.add( new MumFormula(interaction) );
			return answer;
		}
		else if( result.equals("1") ) {
			// korrekte Antwort ungefaehrlich
			answer.add( interaction );
		}
		else if( result.equals("0") ) {
			// Anfrage gilt nicht -> negieren
			Formula neg_formula = new Formula( interaction );
			neg_formula.negate();
			answer.add( neg_formula );
		}
		else {
			throw new DatabaseException("Illegal reaction value.", null);
		}
		
		return answer;
	}
	
	/**
	 * Initiert die Ausloesung des Triggers, welcher die statische signaturbasierte Zugriffskontrolle durchfuehrt.
	 * 
	 * @param CQE_Query: die in ihre einzelnen Teile zerlegte Anfrage
	 * @param interaction: die vom Benutzer eingegebene Anfrage (Interaktion)
	 * @param columnNames: die Namen der Spalten, die in der jeweilig angefragten Relation vorhanden sind
	 * @return String: ist ein potentielles Geheimnis gefaehrdet, wird mum zureck gegeben, andernfalls die korrekte Antwort, kodiert durch 0 (gilt nicht in der DB) bzw. 1 (gilt in der DB)             
	 * @throws DatabaseException, UnsupportedFormulaException, IllegalFormulaOperationException
	 */
	protected abstract String writeToCQEQuery( Client client, ArrayList<String> CQE_Query, Formula interaction, String[] ColumnNames, String relationname ) throws DatabaseException, UnsupportedFormulaException;
}
