package server.censor.refusal;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import communication.CqeType;

import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.ParserException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;

import notification.serverToClient.UpdateResultNotification;

import server.censor.CensorUtil;
import server.censor.UpdateCensor;
import server.core.Client;
import server.core.Server;
import server.data.ConfidentialityPolicyEntry;
import server.data.IdAndFormulaEntry;
import server.data.Log;
import server.data.LogEntry;
import server.parser.Formula;
import server.parser.MumFormula;
import server.parser.node.AtomPredicateNode;
import server.parser.node.ConjunctorNode;
import server.parser.node.NegationNode;
import server.parser.node.Node;
import server.theoremprover.ProverNine;
import server.theoremprover.TheoremProver;
import user.IdAndFormula;

public class RefusalUpdateCensor extends UpdateCensor {
	
	public static final String ANSWER_REFUSED = "transaction refused";
	public static final String ANSWER_VIOLATES_POLICY = "transaction violates confidentiality or user knows integrity violation";
	public static final String ANSWER_VIOLATES_INTEGRITY = "transaction violates integrity";
	public static final String ANSWER_INTEGRITY_VIOLATES_POLICY = "integrity check conflicts confidentiality";
	public static final String ANSWER_SUCCESS = "transaction successful";
	
	
	private TheoremProver tp = new ProverNine();
	private RefusalContinuousCensor queryCensor;

	public RefusalUpdateCensor( Server server ) {
		super( "RefusalUpdate", server );
		this.queryCensor = new RefusalContinuousCensor( server );
	}

	@Override
	public UpdateResultNotification ptr(Client client, List<Formula> literals) {
		return new UpdateResultNotification( "Bisher nicht implementiert", "", new ArrayList<IdAndFormula>() );
	}

	@Override
	public UpdateResultNotification pup(Client client, Formula literal) {
		return new UpdateResultNotification( "Bisher nicht implementiert", "", new ArrayList<IdAndFormula>() );
	}

	@Override
	public UpdateResultNotification vtr(Client client, List<Formula> literals) throws Exception {
		List<IdAndFormula> addToLog = new LinkedList<IdAndFormula>();
		
		// Interaction als String speichern. Wird fuers Log benoetigt.
		Formula interaction = new Formula("VTR("+listToString(literals)+")");
		interaction.setInteractionType( CqeType.InteractionType.VIEW_UPDATE_TRANSACTION );
		
		// Fall 1: Ausstehende Updates
		for ( Formula literal : literals ) {
			// Anfrage nach dem Query-Protokoll abarbeiten. Das Ergebnis steht danach im Log des Benutzers.
			IdAndFormula result = null;
			try {
				result = this.queryCensor.evaluate(client, literal).getResult().get(0);
				addToLog.add( result );
				
				if ( result.getFormula() instanceof MumFormula ) {
					return new UpdateResultNotification( ANSWER_REFUSED, literal.toString(), addToLog );
				}
			} catch (ParserException e) {
				// Kann niemals auftreten.
				e.printStackTrace();
			} catch (DictionaryTooSmallException e) {
				// Kann niemals auftreten.
				e.printStackTrace();
			}
		}

		// Berechne ausstehende Updates
		List<Formula> incDeltaLiterals = new ArrayList<Formula>();
		List<AtomPredicateNode> incDelta = new ArrayList<AtomPredicateNode>();
		Set<AtomPredicateNode> incDeltaSet = new HashSet<AtomPredicateNode>();
		for ( Formula literal : literals ) {
			if ( client.getApplicationDB().evalComplete(literal) == false ) {
				// In incDeltaLiterals kommen auch Negationen mit rein, um sie zum Update verwenden
				// zu koennen.
				incDeltaLiterals.add( literal );

				// Wir haben die Garantie, dass literal immer eine NegationNode mit AtomPredicateNode als Kind oder
				// direkt eine AtomPredicateNode ist.
				AtomPredicateNode atom = null;
				if ( literal.getRootChild().isNegationNode() ) {
					atom = (AtomPredicateNode) ((NegationNode)literal.getRootChild()).getNegatedFormula();
				} else {
					atom = (AtomPredicateNode) literal.getRootChild();
				}
				
				incDelta.add( atom );
				incDeltaSet.add( atom );
			}
		}
		
		
		// Log, PriorUser, PriorAll und Constraints zusammenfassen.
		List<Formula> view = new ArrayList<Formula>();
		for ( LogEntry logEntry : client.getUser().getLog() ) {
			view.add( logEntry.getAnswer() );
		}
		for ( IdAndFormulaEntry entry : client.getUser().getPriorUser() ) {
			view.add( entry.getFormula() );
		}
		for ( IdAndFormulaEntry entry : client.getMaintenanceDB().getPriorAll() ) {
			view.add( entry.getFormula() );
		}
		for ( IdAndFormulaEntry entry : client.getMaintenanceDB().getSchemaConstraints() ) {
			view.add( entry.getFormula() );
		}
		
		// View kopieren, damit Modifikationen okay sind und danach die effektiven Updates im Log durchfuehren
		// (wird fuer Fall 2 und 3 benoetigt).
		List<Formula> updatedView = new ArrayList<Formula>();
		for ( Formula formula : view ) {
			updatedView.add( new Formula(formula) );
		}
		CensorUtil.variableNegationOnSetsOfFormulas(updatedView, incDelta);
		
		Node constraintConjunction = new ConjunctorNode();
		// neg(view_i-1, Inc_Delta) u con
		List<Node> updatedViewWithConstraints = new ArrayList<Node>( updatedView );
		for ( IdAndFormulaEntry constraint : client.getMaintenanceDB().getSchemaConstraints() ) {
			updatedViewWithConstraints.add( constraint.getFormula() );
			
			// Direkt die Konjunktion der Constraints bilden, wird fuer Fall 3 gebraucht.
			constraintConjunction.addChild( constraint.getFormula() );
		}
		
		// Wenn die Konjunktion der Constraints nur 1 Kind hat, dann nehmen wir direkt das.
		// Wenn die Konjunktion gar kein Kind hat, dann gibt es keine Constraints => null.
		if ( constraintConjunction.getChildrenNumber() == 1 ) {
			constraintConjunction = constraintConjunction.getChild(0);
		} else if ( constraintConjunction.getChildrenNumber() == 0 ) {
			constraintConjunction = null;
		}
		
		
		
		// Effektive Updates (SDeltaStrich) berechnen.
		List<Set<AtomPredicateNode>> SDelta = this.queryCensor.calculateLastSDelta( queryCensor.getUpdatesFromLog(client.getUser().getLog()) );
		List<Set<AtomPredicateNode>> SDeltaDash = this.queryCensor.calculateNextSDelta(SDelta, incDeltaSet);
		
		logger.debug("Inc_Delta: "+incDeltaSet);
		logger.debug("S_Delta: "+SDelta);
		logger.debug("S_Delta Strich: "+SDeltaDash);
		
		// Fall 2: Das Ausfuehren der Transaktion wuerde ein Geheimnis verraten
		for ( ConfidentialityPolicyEntry confPolItem : client.getUser().getConfidentialityPolicy() ) {
			Formula psi = confPolItem.getFormula();
			
			// Wir unterstuetzen nur potential secrets.
			if ( confPolItem.getType() != CqeType.PolicyType.POTENTIAL_SECRETS ) {
				throw new UnsupportedConfidentialityPolicyException(confPolItem.getFormula());
			}
			
			// CCP: statt nur das confPolItem psi zu testen wird eine Disjunktion ueber alle
			// effektiven Update-Zeitpunkte getestet (ccp(phi, SDelta-Strich)).
			if ( confPolItem.getPreservation() == CqeType.PolicyPreservation.CONTINUOUS ) {
				psi = this.queryCensor.ccp( confPolItem, SDeltaDash );
			}
			
			logger.debug("psi: "+confPolItem);
			logger.debug("ccp(psi, SDeltaStrich): "+psi);
						
			// neg(view, IncDelta) u con |= ccp(psi, SDeltaStrich) fuer CCP,
			// neg(view, IncDelta) u con |= psi für nicht CCP
			if ( this.tp.prove(updatedViewWithConstraints, psi, false) ) {
				return new UpdateResultNotification(ANSWER_VIOLATES_POLICY, literals.toString(), addToLog );
			}
		}
		
		
		
		// Fall 3: Integritaetspruefung.
		if ( constraintConjunction != null ) {
			if ( this.tp.prove(updatedView, constraintConjunction, false) == false ) {
				// Constraints werden verletzt.
				// constraintConjunction wird nicht mehr benoetigt, kann deshalb manipuliert werden.
				
				// Bilde neg( NOT constraintConjunction, IncDelta)
				Formula negatedUpdatedConstraintConjunction = new Formula(constraintConjunction);
				negatedUpdatedConstraintConjunction.negate();
				negatedUpdatedConstraintConjunction.neg( incDelta );
				
				// Unterfall 1: Verletzung ist bekannt.
				if ( this.tp.prove(view, negatedUpdatedConstraintConjunction, false) ) {
					return new UpdateResultNotification(ANSWER_VIOLATES_INTEGRITY, literals.toString(), addToLog );
				}
				
				// Unterfall 2: Benachrichtigung ueber Verletzung wuerde Geheimnis in Gefahr bringen.
				List<Formula> viewAndNegatedUpdatedConstraintConjunction = new ArrayList<Formula>( view );
				viewAndNegatedUpdatedConstraintConjunction.add( negatedUpdatedConstraintConjunction );
				
				for ( ConfidentialityPolicyEntry confPolItem : client.getUser().getConfidentialityPolicy() ) {
					Formula psi = confPolItem.getFormula();
					
					// CCP: statt nur das confPolItem psi zu testen wird eine Disjunktion ueber alle
					// effektiven Update-Zeitpunkte getestet (ccp(phi, SDelta-Strich)).
					if ( confPolItem.getPreservation() == CqeType.PolicyPreservation.CONTINUOUS ) {
						psi = this.queryCensor.ccp( confPolItem, SDelta );
					}
					
					// view u neg( NOT con_conj, IncDelta) |= ccp(psi, SDeltaStrich) bzw. nur psi
					if ( this.tp.prove(viewAndNegatedUpdatedConstraintConjunction, psi, false) ) {
						return new UpdateResultNotification(ANSWER_INTEGRITY_VIOLATES_POLICY, literals.toString(), addToLog );
					}
				}
				
				// Unterfall 3: Integritaetscheck fehlgeschlagen: pruefe auf eval(neg( NOT con_conj, IncDelta ))(db_i-1) == true
				// View wird automatisch von evalComplete() aktualisiert.
				if ( client.getApplicationDB().evalComplete(negatedUpdatedConstraintConjunction) == true ) {
					return new UpdateResultNotification(ANSWER_VIOLATES_INTEGRITY, literals.toString(), addToLog );
				}
			}
		}
		

		
		// Fall 4: Update erfolgreich.
		
		// Aenderung in der Datenbank durchfuehren.
		int updateCount = client.getApplicationDB().updateComplete( incDeltaLiterals );
		if ( updateCount != incDeltaLiterals.size() ) {
			throw new DatabaseException( "Unable to update all literals in incDelta. Update count was "+updateCount+" but should be"+incDeltaLiterals.size(), null );
		}
		
		// Log aktualisieren.
		this.updateLog( client.getUser().getLog(), incDelta );
		
		// Update ins Log eintragen.
		client.getUser().getLog().add(interaction, null, null, new HashSet<AtomPredicateNode>(incDelta));
		
		// FIXME: Log muss ggf. ausgetauscht werden..
		return new UpdateResultNotification( ANSWER_SUCCESS, "", addToLog );
	}

	@Override
	public UpdateResultNotification vup(Client client, Formula literal) {
		return new UpdateResultNotification( "Bisher nicht implementiert", "", new ArrayList<IdAndFormula>() );
	}
	
	private void updateLog( Log log, List<AtomPredicateNode> incDelta ) throws UnsupportedFormulaException, DatabaseException {
		for ( LogEntry entry : log ) {
			Formula copy = entry.getAnswer();
			copy.neg( incDelta );
			entry.setAnswer(copy);
		}
	}

	/**
	 * Translates a list of objects into a comma seperated String surrounded by angle brackets.
	 * 
	 * @param list The list that will be translated.
	 * @return A String with a representation of all objects of the list seperated by commas.
	 */
	public static String listToString( List<?> list ) {
		String result = "";
		
		String sep = "";
		for ( Object obj : list ) {
			result += sep + obj.toString();
			sep = ",";
		}
		
		return result;
	}
}
