package server.censor.refusal;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import communication.CqeType;

import server.core.Client;
import server.core.Server;
import server.data.ConfidentialityPolicyEntry;
import server.data.Log;
import server.data.LogEntry;
import server.parser.Formula;
import server.parser.MumFormula;
import server.parser.node.AtomPredicateNode;
import server.parser.node.DisjunctorNode;
import server.theoremprover.ProverNine;
import server.theoremprover.TheoremProver;
import user.IdAndFormula;
import exception.DatabaseException;
import exception.ProverException;
import exception.UnsupportedConfidentialityPolicyException;
import exception.UnsupportedFormulaException;

/**
 * Ist von der Arbeitsweise mit dem Refusal Censor identisch, unterscheidet allerdings
 * zwischen continuous und temporary confidentiality preservation.
 * 
 * Bei temporary confidentiality preservation entspricht das Verhalten exakt dem Refusal Censor.
 * 
 * Implementierung nach Protocol 1 aus Abschnitt 6.1 "Inference-proof View Update Transactions with
 * Minimal Refusals" von Joachim Biskup und Cornelia Tadros.
 */
public class RefusalContinuousCensor extends RefusalCensor {
	
	private TheoremProver tp = new ProverNine();

	public RefusalContinuousCensor( Server server ) {
		super( server );
	}
	
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws ProverException, UnsupportedFormulaException, DatabaseException, UnsupportedConfidentialityPolicyException {
		List<Formula> result = new LinkedList<Formula>();
		result.add( this.modify(client, interaction) );
		return result;
	}
	
	/**
	 * Berechnet die Anfrageauswertung fuer den Refusal-Zensor, der auch zwischen continuous und temporary confidentiality
	 * preservation unterscheidet.
	 * 
	 * Der Zensor benoetigt eine vollstaendige Datenbankinstanz, ausschliesslich potential secrets in der confidentiality
	 * policy und geschlossene Anfragen.
	 * 
	 * Das genaue Verfahren ist in der Arbeit "Inference-proof View Update Transactions with Minimal Refusals" von
	 * Joachim Biskup und Cornelia Tadros aus dem Jahr 2011 beschrieben.
	 * 
	 * @param client
	 * @param interaction
	 * @return
	 * @throws ProverException 
	 * @throws UnsupportedConfidentialityPolicyException 
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 */
	private Formula modify(Client client, Formula interaction) throws ProverException, UnsupportedConfidentialityPolicyException, UnsupportedFormulaException, DatabaseException {
		List<IdAndFormula> log = client.getUser().getKnowledge().getCopyAsList();
		List<Set<AtomPredicateNode>> SDelta = this.calculateLastSDelta( this.getUpdatesFromLog(client.getUser().getLog()) );
		
		logger.debug(log);
		
		Formula negatedInteraction = new Formula(interaction.clone());
		negatedInteraction.negate();
		
		// Fall 1: Antwort dem Benutzer bekannt.
		// Teste ob log |= phi oder log |= NICHT phi. Dann weiss der Benutzer das Ergebnis der Anfrage
		// bereits und wir koennen ihm sein bisheriges Wissen mitteilen.
		if ( super.isFormulaImpliedByLog(interaction, log) ) {
			return interaction;
		}
		
		if ( super.isFormulaImpliedByLog(negatedInteraction, log) ) {
			return interaction;
		}
		
		// Vereinigung aus Log und (negierter) Anfrage erstellen.
		LinkedList<Formula> logAndQuery = new LinkedList<Formula>();
		LinkedList<Formula> logAndNegatedQuery = new LinkedList<Formula>();
		
		for ( IdAndFormula logEntry : log ) {
			logAndQuery.add( logEntry.getFormula() );
			logAndNegatedQuery.add( logEntry.getFormula() );
		}
		logAndQuery.add( interaction );
		logAndNegatedQuery.add( negatedInteraction );
		
		// Fall 2: Antwort wuerde ein Geheimnis verraten.
		for ( ConfidentialityPolicyEntry entry : client.getUser().getConfidentialityPolicy() ) {
			Formula psi = entry.getFormula();
			
			// Wir unterstuetzen nur potential secrets.
			if ( entry.getType() != CqeType.PolicyType.POTENTIAL_SECRETS ) {
				throw new UnsupportedConfidentialityPolicyException(entry.getFormula());
			}
			
			// CCP: statt nur das potSecItem psi zu testen wird eine Disjunktion ueber alle
			// Update-Zeitpunkte getestet (ccp(phi)).
			if ( entry.getPreservation() == CqeType.PolicyPreservation.CONTINUOUS ) {
				psi = ccp( entry, SDelta );
			}
			
			// log u phi |= psi bzw. log u phi |= ccp(psi) (je nach Fall).
			if ( this.tp.prove(logAndQuery, psi, true) ) {
				return new MumFormula();
			}
			// log u NOT phi |= psi bzw. log u NOT phi |= ccp(psi) (je nach Fall).
			if ( this.tp.prove(logAndNegatedQuery, psi, true) ) {
				return new MumFormula();
			}
		}
		
		// Fall 3: Anfrage bzgl. der Datenbank beantworten.
		return Formula.createFormula( client.getApplicationDB().evalStarComplete(interaction) );
	}
	
	/**
	 * Implementiert die ccp() Methode beschrieben in Definition 4 (Kapitel 5).
	 * Erzeugt eine Disjunktion ueber die aktualisierte Update-Historie.
	 * 
	 * @param potSecElement
	 * @param updateDifferences
	 * @return
	 * @throws UnsupportedConfidentialityPolicyException
	 * @throws UnsupportedFormulaException
	 */
	public Formula ccp( ConfidentialityPolicyEntry potSecElement, List<Set<AtomPredicateNode>> updateDifferences ) throws UnsupportedConfidentialityPolicyException, UnsupportedFormulaException {
		// Diese Methode funktioniert nur mit potential secrets (nicht mit secrecies), die
		// dauerhaft geschuetzt werden sollen (continuous confidentiality).
		if ( potSecElement.getType() != CqeType.PolicyType.POTENTIAL_SECRETS || potSecElement.getPreservation() != CqeType.PolicyPreservation.CONTINUOUS ) {
			throw new UnsupportedConfidentialityPolicyException(potSecElement.getFormula());
		}
		
		// Kopie des potsec Elements anlegen, damit Veraenderungen die Eingabe nicht beeinflussen.
		Formula phi = (Formula)potSecElement.getFormula().clone();
		
		// Wenn updateHistory leer ist, dann direkt das potSecElement zurueckgeben.
		if ( updateDifferences.isEmpty() ) {
			return phi;
		}
		
		// Disjunktion ueber alle neg(potSecElement, updateHistory[i]) aufbaun.
		DisjunctorNode disjunction = new DisjunctorNode();
		for ( Set<AtomPredicateNode> effectiveUpdate : updateDifferences ) {
			// phi jedes mal neu kopieren und aendern.
			phi = (Formula)potSecElement.getFormula().clone();
			phi.neg( effectiveUpdate );
			disjunction.addChild( phi );
		}
		// Als letztes phi selbst hinzufuegen.
		phi = (Formula)potSecElement.getFormula().clone();
		disjunction.addChild( phi );
		
		return new Formula( disjunction );
	}
	
	/**
	 * Berechnet anhand der Update-Historie (alle effektiven Updates) das aktuelle
	 * SDelta_i.
	 * 
	 * @param updateHistory
	 * @return
	 */
	public List<Set<AtomPredicateNode>> calculateLastSDelta( List<Set<AtomPredicateNode>> updateHistory ) {
		List<Set<AtomPredicateNode>> result = new ArrayList<Set<AtomPredicateNode>>();
		
		for ( Set<AtomPredicateNode> effectiveUpdate : updateHistory ) {
			result = this.calculateNextSDelta( result, effectiveUpdate );
		}
		
		return result;
	}
	
	/**
	 * Berechnet zu einem gegebenen SDelta_i-1 und neue effektive Updates das neue SDelta_i.
	 * 
	 * @param oldSDelta
	 * @param effectiveUpdate
	 * @return
	 */
	public List<Set<AtomPredicateNode>> calculateNextSDelta( List<Set<AtomPredicateNode>> oldSDelta, Set<AtomPredicateNode> effectiveUpdate ) {
		List<Set<AtomPredicateNode>> result = new ArrayList<Set<AtomPredicateNode>>();
		
		for ( Set<AtomPredicateNode> delta : oldSDelta ) {
			// SDelta[s] \ Var(IncDelta)
			Set<AtomPredicateNode> part1 = new HashSet<AtomPredicateNode>(delta);
			part1.removeAll(effectiveUpdate);
			// Var(IncDelta) \ SDelta[s]
			Set<AtomPredicateNode> part2 = new HashSet<AtomPredicateNode>(effectiveUpdate);
			part2.removeAll(delta);
			
			// (SDelta[s] \ Var(IncDelta)) u (Var(IncDelta) \ SDelta[s])
			Set<AtomPredicateNode> newDelta = new HashSet<AtomPredicateNode>();
			newDelta.addAll( part1 );
			newDelta.addAll( part2 );
			
			result.add( newDelta );
		}
		
		// SDelta[i-1] (also letztes neues Element) sind genau die ausstehenden Updates.
		result.add( effectiveUpdate );
		
		return result;
	}
	
	/**
	 * Get all previous updates from the user's log.
	 * @param log
	 * @return
	 */
	public List<Set<AtomPredicateNode>> getUpdatesFromLog( Log log ) {
		List<Set<AtomPredicateNode>> result = new LinkedList<Set<AtomPredicateNode>>();
		
		for ( LogEntry entry : log ) {
			// FIXME: check for update types and not non-query type.
			if( entry.getInteractionType().compareToIgnoreCase("QUERY") != 0 ) {
				result.add( entry.getEffectiveUpdates() );
			}
		}
		
		return result;
	}

	@Override
	public String getName() {
		return "RefusalContinuous";
	}

}
