package server.censor.lying;

import java.util.LinkedList;
import java.util.List;

import communication.CqeType;

import server.censor.CensorUtil;
import server.censor.QueryCensor;
import server.core.Client;
import server.core.Server;
import server.parser.Formula;
import server.parser.FormulaUtil;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;

import exception.DatabaseException;
import exception.ProverException;
import exception.UnsupportedFormulaException;

public class LyingCensor extends QueryCensor {

	public LyingCensor(Server server) {
		super("Lying", server);
	}
	
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws DatabaseException, UnsupportedFormulaException, ProverException {
		return this.run(client, interaction, client.getUser().getKnowledge().getCopyAsList());
	}
	
	/**
	 * Die Methode implementiert die (dynamische) kontrollierte Anfrageauswertung mit dem Luegen-Zensor bei bekannten potentiellen Geheimnissen
	 * und einer als vollstaendig angenommenen Datenbankinstanz. Falls durch die korrekte Antwort auf die gestellte Anfrage die
	 * Sicherheitspolitik verletzt wird (die Disjunktion aller potentiellen Geheimnisse kann erschlossen werden), wird die Negation
	 * der korrekten Antwort zurueck gegeben. Falls durch die korrekte Antwort die Sicherheitspolitik hingegen nicht verletzt wird,
	 * wird die korrekte Antwort zurueck gegeben.
	 * Diese Methode mit dem zusaetzlichen Parameter log wird vom den OpenLyingCensor benoetigt.
	 *  
	 * @param client
	 * @param interaction Anfrage des Clients.
	 * @param log og, der fuer die Auswertung der Anfrage verwendet werden soll (wichtig fuer den OpenlyingCensor)
	 * @return
	 * @throws UnsupportedFormulaException 
	 * @throws DatabaseException 
	 * @throws ProverException 
	 */
	protected List<Formula> run(Client client, Formula interaction, List<IdAndFormula> log ) throws DatabaseException, UnsupportedFormulaException, ProverException {
		boolean answer = false;
		List<Formula> result = new LinkedList<Formula>();

		List<ConfidentialityPolicyItem> confidentialityPolicy = client.getUser().getConfidentialityPolicy().getCopyAsList();

		// Eval-Stern ausfuehren
		Formula evalStarQuery = Formula.createFormula(client.getApplicationDB().evalStarComplete(interaction));

		answer = this.isConfidentialityPolicyViolated( evalStarQuery, log, confidentialityPolicy );
		logger.debug( "Lügen-Zensor isConfidentialityPolicyViolated(): "+ answer );

		// Luegen, falls korrekte Antwort gefaehrlich
		if ( answer == true ) {
			evalStarQuery.negate();
		}

		// Antwort wird konstruiert: entweder korrekte Antwort oder Luege

		result.add( evalStarQuery );		

		return result;		
	}

	/**
	 * Fuer die Formelmenge log u query wird geprueft, ob durch diese die Disjunktion der potentiellen Geheimnisse
	 * aus der Sicherheitspolitik impliziert wird.
	 * @param query Ein Query, fuer den geprueft werden soll, ob er zusammen mit dem Log die confidentiality policy verletzt
	 * @param log Das Vorwissen des Benutzers
	 * @param confidentialityPolicy Die Sicherheitspolitik in Form von potentiellen Geheimnissen
	 * @return true, falls durch log u query die Disjunktion der potentiellen Geheimnisse impliziert wird; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 */
	protected boolean isConfidentialityPolicyViolated( Formula query, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy ) throws ProverException {
		List<Formula> querys = new LinkedList<Formula>();
		querys.add(query);
		
		return this.isConfidentialityPolicyViolated(querys, log, confidentialityPolicy);
	}
	
	/**
	 * Fuer die Formelmenge log u querys wird geprueft, ob durch diese die Disjunktion der potentiellen Geheimnisse
	 * aus der Sicherheitspolitik impliziert wird.
	 * @param querys Eine Menge von Querys, fuer die geprueft werden soll, ob sie zusammen mit dem Log die confidentiality policy verletzen
	 * @param log Das Vorwissen des Benutzers
	 * @param confidentialityPolicy Die Sicherheitspolitik in Form von potentiellen Geheimnissen
	 * @return true, falls durch log u querys die Disjunktion der potentiellen Geheimnisse impliziert wird; ansonsten wird false zurueck gegeben
	 * @throws ProverException Wird geworfen, falls im Rahmen des Theorembeweiser-Aufrufs ein Fehler auftritt
	 */
	protected boolean isConfidentialityPolicyViolated( List<Formula> querys, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy ) throws ProverException {
//		if( confidentialityPolicy.isEmpty() ) {
//			// eine leere confidentiality policy wird trivialerweise nicht verletzt
//			return false;
//		}
		
		// Disjunktion der Elemente aus confidentialityPolicy erstellen
		Formula pot_sec_disj = this.createDisjunction( confidentialityPolicy );
		
		// Disjunktion in einen neuen IdAndFormula-Vector einfuegen
		confidentialityPolicy = new LinkedList<ConfidentialityPolicyItem>();
		confidentialityPolicy.add( new ConfidentialityPolicyItem(0, pot_sec_disj, CqeType.PolicyType.POTENTIAL_SECRETS, CqeType.PolicyPreservation.CONTINUOUS) );
		
		// Implikation mit Hilfe der Methode der Oberklasse Censor pruefen
		return CensorUtil.isConfidentialityPolicyViolated( querys, log, confidentialityPolicy );
	}

	/**
	 * Dient als Hilfsmethode, um die Disjunktion pot_sec_disj der Elemente der confidentialityPolicy zu bilden.
	 * @return Die Disjunktion pot_sec_disj der Elemente der confidentialityPolicy.
	 * @param confidentialityPolicy Die Sicherheitspolitik, ueber deren Elemente die Disjunktion gebildet werden soll.
	 */
	Formula createDisjunction (List<ConfidentialityPolicyItem> confidentialityPolicy) {
		
		List<Formula> formulas = new LinkedList<Formula>();
		
		for ( ConfidentialityPolicyItem policyItem : confidentialityPolicy ) {
			formulas.add((Formula)policyItem.getFormula().clone() );
		}
		
		return FormulaUtil.createDisjunction(formulas);
	}
}
