package server.censor.combined;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import exception.DatabaseException;
import exception.DictionaryTooSmallException;
import exception.ProverException;
import exception.UnsupportedFormulaException;
import server.censor.CensorUtil;
import server.censor.OpenCensor;
import server.core.Client;
import server.core.Server;
import server.parser.CompletenessSentenceFormula;
import server.parser.Formula;
import server.parser.MumFormula;
import user.ConfidentialityPolicyItem;
import user.IdAndFormula;


/**
 * Der OpenCombinedCensor behandelt offene Anfragen.
 * 
 * @author bring
 *
 */
public class OpenCombinedCensor extends OpenCensor {
	
	protected CombinedCensor closedCombinedCensor;
	
	
	/**
	 * Konstruktur: Erzeugt ein neues Objekt dieses Zensor-Typs
	 * @param appDB App-Datenbank
	 * @param maintenanceDB Maintenance-Datenbank
	 * @param user Der User, fuer den ein konkretes Objekt dieses Zensors Entscheidungen treffen soll.
	 */
	public OpenCombinedCensor(Server server) {
		super("OpenCombined", server);
	}

	
	/** 
	 *  Die Methode implementiert die (dynamische) kontrollierte Anfrageauswertung 
	 *  mit dem kombinierten Zensor für offene Anfragen bei bekannten potentiellen 
	 *  Geheimnissen und einer als vollstaendig angenommenen Datenbankinstanz.
	 *  Die einzelnen Substitutionen der offenen Variable durch Konstanten werden
	 *  dabei jeweils durch den CombinedCenser für geschlossene Anfragen ausgewertet
	 *  und dessen Auswertung bei Verletzung von Elementen aus pot_sec entsprechend
	 *  modifiziert.
	 *  @param query Die an die Datenbank gestellte offene Anfrage
	 */
	@Override
	protected List<Formula> run(Client client, Formula interaction) throws DatabaseException, ProverException, UnsupportedFormulaException, DictionaryTooSmallException {
		// Referenz auf CombinedCensor für geschlossene Anfragen erstellen, falls noch nicht geschehen
		if( closedCombinedCensor == null )
			closedCombinedCensor = (CombinedCensor)this.getServer().getCensorManagement().getCensor("Combined");
		
		// keine offene Anfrage?
		if ( ! interaction.isOpenFormula() )
			throw new UnsupportedFormulaException(interaction.toString() + " ist keine offene Anfrage");
		
		// freie Variablen ermitteln
		Set<String> freeVars = interaction.getFreeVariables();
		
		// benoetigtes Dictionary laden
		Map<String, List<String>> dictionary = this.fillDictionary(client, interaction, freeVars);
		
		
		List<Formula> result = new LinkedList<Formula>();
		// log und PotSec-Menge aus dem User-Objekt laden
		List<ConfidentialityPolicyItem> confidentialityPolicy = client.getUser().getConfidentialityPolicy().getCopyAsList();
		List<IdAndFormula> log = client.getUser().getKnowledge().getCopyAsList();
		
		// Vector fuer die Substitutionen, die im endgueltigen Completenesssentence enthalten sind
		// Dieser ist noetig, weil der finale CS auch alle negierten Antworten abdecken soll, es
		// muessen somit nur die Substitutionen positiver Antworten ausgeschlossen werden
		Set<Map<String, String>> finalCompletenessSentenceSubstitutions = new HashSet<Map<String,String>>();
		
		
		// Phase 1 [determine an appropriate completeness sentence together with a last positive answer]
		
		
		// Bestimme k = Min{ i | Complete(phi(x), i) is true in db }
		int k = getLastConstantSatisfyingPhi(client, dictionary, interaction);
		logger.debug("k=" + k);
		
		// [forward searching] determine a minimal m>=k
		// such that for all psi in pot_sec: log u {Complete(phi(x), m)} not_implies psi
		int m = determineM(client, dictionary, interaction, log, confidentialityPolicy, k);
		logger.debug("m=" + m);
		Formula complete_m = this.generateCompletenessSentence(client, dictionary, interaction, m);
		
		
		// [backward searching] determine k* = Max{ j | j<=m and for all psi in pot_sec: log u {phi(c_j), Complete(phi(x), m)} not_implies psi }
		int kStar = m;
		if( kStar > 0 ) {	// andernfalls gibt es keine positive Antwort
			List<Formula> sos = new LinkedList<Formula>();
			Map<String, String> mapping;
			mapping = this.getFreeVarMapping(dictionary, freeVars, kStar);
			Formula closedQuery = new Formula(interaction);
			closedQuery.substituteVariables(mapping);
			sos.add( closedQuery );
			sos.add( complete_m );
			while( CensorUtil.isConfidentialityPolicyViolated(sos, log, confidentialityPolicy) ) {
				--kStar;
				if(kStar == 0) {
					// es existiert keine letzte positive Antwort
					// k* ist in diesem Fall 0 und die Schleife darf verlassen werden, da nur der
					// Vollstaendigkeitssatz alleine bereits im vorherigen Schritt als harmlos eingestuft wurde
					break;
				}
				sos.clear();
				mapping = this.getFreeVarMapping(dictionary, freeVars, kStar);
				closedQuery = new Formula(interaction);
				closedQuery.substituteVariables(mapping);
				sos.add(closedQuery);
				sos.add(complete_m);
			}
		}
		logger.debug("k*=" + kStar);
		
		
		// [forward searching] determine m* = Min{ j | k*<=j and for all psi in pot_sec: log u {phi(c_k*), Complete(phi(x), j)} not_implies psi }
		int mStar = kStar;
		List<Formula> sos = new LinkedList<Formula>();
		CompletenessSentenceFormula complete_mStar = this.generateCompletenessSentence(client, dictionary, interaction, mStar);
		sos.add(complete_mStar);
		Map<String, String> mapping;
		Formula closedQuery = null;
		if( kStar > 0 ) {	// fuer k*=0 gibt es keine letzte postive Antwort
			mapping = this.getFreeVarMapping( dictionary, freeVars, kStar );
			closedQuery = new Formula( interaction );
			closedQuery.substituteVariables( mapping );
			sos.add( closedQuery );
		}
		while( CensorUtil.isConfidentialityPolicyViolated( sos, log, confidentialityPolicy ) ) {
			++mStar;
			complete_mStar = this.generateCompletenessSentence( client, dictionary, interaction, mStar );
			sos.clear();
			sos.add( complete_mStar );
			if( kStar > 0 )
				sos.add( closedQuery );
		}
		logger.debug("m*=" + mStar);
		
		// add phi(c_k*) and Complete(phi(x), m*) to answer set and log
		if( kStar > 0 ) {
			// nur wenn es auch eine positive Antwort gibt...
			log.add(new IdAndFormula(0, closedQuery));
			result.add(closedQuery);
			// Substitution in finalem Completenesssentence ausschliessen
			finalCompletenessSentenceSubstitutions.add( this.getFreeVarMapping(dictionary, freeVars, kStar) );
		}
		// Vorlaeufigen CompletenessSentence ins temporaere Log aufnehmen, aber nicht ins Result, da dieser
		// spaeter durch den finalen CompletenessSentence, der auch alle negativen Antworten abdeckt ersetzt wir.
		log.add( new IdAndFormula(0, complete_mStar) );
		
		
		// Phase 2 [inspect ordinary query result]
		for(int j=1; j<=mStar; ++j) {
			if( j == kStar ) {
				// phi(k*) ist bereits in der Antwortmenge und im Log
				continue;
			}
			// behandle phi(c_j) als geschlossenen Query mit dem kombinierten Zensor fuer geschlossene Anfragen
			mapping = this.getFreeVarMapping(dictionary, freeVars, j);
			closedQuery = new Formula(interaction);
			closedQuery.substituteVariables(mapping);
			Formula censorAnswer = closedCombinedCensor.run( client, new Formula(closedQuery), log ).get(0);
			if ( censorAnswer instanceof MumFormula ) {
				// verweigerte Antwort
				// Substitution in finalem CompletenessSentence ausschliessen
				finalCompletenessSentenceSubstitutions.add( mapping );
				// verweigerte Antwort zum Result hinzufuegen
				result.add( censorAnswer );
			}
			else if( censorAnswer.equals(closedQuery) ) {
				// positive Antwort
				// Substitution in finalem CompletenessSentence ausschliessen
				finalCompletenessSentenceSubstitutions.add( mapping );
				// Antwort zum Result und Log hinzufuegen
				result.add( censorAnswer );
				log.add( new IdAndFormula(0, censorAnswer) );
			}
			else {
				// negative Antwort
				// Kombination aus dem Completeness Sentence herausnehmen, sodass dieser die
				// negative Antwort interactionmit abdeckt
				// eine Aufnahme der Antwort ins Log und Result ist daher nicht notwendig, da
				// der Completeness Sentence am Ende sowohl ins Log als auch Result aufgenommen wird
				// Da die Formel complete_mStar im Log enthalten ist (selbes Objekt!), wirkt sich die
				// folgende Modifikation von complete_mStar auch auf die im Log enthaltene Formel aus.
				// Deshalb ist ein manuelles Einfuegen der negativen Antwort ins Log _nicht_ erforderlich.
				complete_mStar.removeSubstitution( mapping );
			}
		}
		
		// finalen CompletenessSentence aufbauen und zum Ergebnis hinzufuegen
		CompletenessSentenceFormula finalCompletenessSentence = new CompletenessSentenceFormula( interaction, finalCompletenessSentenceSubstitutions );
		result.add( finalCompletenessSentence );
			
		return result;
	}
	
	/**
	 * Diese Methode berechnet den kleinsten Index m>=k, sodass 
	 * log vereinigt Complete(phi, m) not_implies psi (fuer jedes psi aus pot_sec) gilt.
	 * 
	 * @param interaction	Formel psi bzgl. der die Berechnung durchgefuehrt werden soll
	 * @param log			log des Benutzers
	 * @param confidentialityPolicy Confidentiality Policy
	 * @param k				kleinster Index, fuer den Complete(phi, k) in der Datenbank wahr ist
	 * @return     Index m (s.o.)
	 * @throws DictionaryTooSmallException
	 * @throws ProverException
	 * @throws DatabaseException
	 */
	private int determineM(Client client, Map<String, List<String>> dictionary, Formula interaction, List<IdAndFormula> log, List<ConfidentialityPolicyItem> confidentialityPolicy, int k) throws DatabaseException, UnsupportedFormulaException, DictionaryTooSmallException, ProverException {
		// binaere Suche
		// suche Index i=2^j fuer den log vereinigt CS(phi, k+i) erstmalig kein Geheimnis impliziert
		int i = 1;
		Formula complete_m = this.generateCompletenessSentence(client, dictionary, interaction, k);
		while( CensorUtil.isConfidentialityPolicyViolated(complete_m, log, confidentialityPolicy) ) {
			i *= 2;
			complete_m = this.generateCompletenessSentence(client, dictionary, interaction, k+i);
		}
		// m enthaelt den bisher kleinsten bisher gefundenen Wert fuer den log vereinigt CS(phi, m) kein Geheimnis impliziert
		int m = k+i;
		
		// fuer i=2^(j-1) gilt log vereinigt CS(phi, m) impliziert kein Geheimnis nicht
		// fuer i=2^j gilt log vereinigt CS(phi, m) impliziert kein Geheimnis
		// -> binare Suche im Intervall von k+i/2(=k + 2^(j-1)) bis k+i-1(=k + 2^j - 1)
		int left = k + i/2;
		int right = k + i-1;
		while( left <= right ) {
			int middle = (left + right) / 2;
			complete_m = this.generateCompletenessSentence(client, dictionary, interaction, middle);
			if( ! CensorUtil.isConfidentialityPolicyViolated(complete_m, log, confidentialityPolicy) ) {
				// kleineren Wert fuer m gefunden
				m = middle;
				// links nach noch kleineren Werten weitersuchen
				right = middle - 1;
			}
			else {
				// rechts weitersuchen
				left = middle + 1;
			}
		}
		
		return m;
	}
}
