package communication;

/**
 * Sammlung wichtiger ENUM-Typen, die sowohl vom Client als auch Server gebraucht werden.
 */
public class CqeType {

	/**
	 * Rechte werden an unterschiedlichen Stellen im Server ueberprueft. Hat ein Benutzer das
	 * geforderte Recht nicht, so wird ihm der Zugriff verweigert.
	 */
	public enum Right {
		/**
		 * Benutzer hat sich mit Login und Passwort authentifiziert.
		 */
		USER_LOGGED_IN			("logged in"),
		/**
		 * Kann normale Anfragen stellen, die durch einen Zensor auf Inferenzsicherheit geprüft werden.
		 */
		SECURE_INTERACTION		("inference-proof interaction"),
		/**
		 * Kann direkte Anfragen an das Informationssystem stellen, die NICHT auf Inferenzsicherheit geprüft werden.
		 */
		UNSECURE_INTERACTION	("non-inference-proof interaction"),
		/**
		 * Kann die Sicherheitspolitik sehen. Nur wenn dieses Recht gesetzt wird, wird
		 * die Sicherheitspolitik an den Client gesendet.
		 * Achtung: Das Recht MANAGE_USERS erlaubt die Einsicht der Sicherheitspolitiken
		 * aller Benutzer, also auch des eigenen.
		 */
		VIEW_CONFIDENTIALITY_POLICY	("view confidentiality policy"),
		/**
		 * Kann Benutzer anlegen, modifizieren und löschen. Dazu gehört auch die Manipulation des Logs und der
		 * ConfidentialityPolicy. Zusammen mit MANAGE_AUTOMATON kann auch der aktuelle Zustand des Benutzers
		 * manipuliert werden.
		 */
		MANAGE_USERS			("user management"),
		/**
		 * Kann die Constraints für alle Datenbanken verwalten.
		 */
		MANAGE_CONSTRAINTS		("constraint management"),
		/**
		 * Kann den Aufbau des Automaten festlegen. Legt damit gleichzeitig fest, wann welcher Zensor benutzt werden kann.
		 * Zusammen mit MANAGE_USERS kann pro Benutzer festgelegt werden, in welchem Zustand dieser sich gerade befindet.
		 */
		MANAGE_AUTOMATON		("automaton management");
		
		
		
		private final String description;
		Right( String desc) {
			this.description = desc;
		}

		public String getDescription() { return this.description; }
		public String toString() { return this.getDescription(); }
		
		public static Right stringToRight( String name ) { return Enum.valueOf(Right.class, name); }
	}
	
	/**
	 * Der Interaktionstyp einer Formel gibt an, welche Art von Formel kodiert wird.
	 */
	public enum InteractionType {
		/**
		 * Allgemeine praedikatenlogische Formel.
		 */
		NONE,
		/**
		 * Anfrage eines Benutzers.
		 */
		QUERY,
		/**
		 * Einzelne Update-Anfrage eines Benutzers.
		 */
		VIEW_UPDATE,
		/**
		 * Update-Transaktion (mehrere Anfragen) eines Benutzers.
		 */
		VIEW_UPDATE_TRANSACTION,
		/**
		 * Einzelne Update-Anfrage des Administrators.
		 */
		PROVIDER_UPDATE,
		/**
		 * Update-Transaktion (mehrere Anfragen) des Administrators.
		 */
		PROVIDER_UPDATE_TRANSACTION
	}
	
	/**
	 * Art der Sicherheitspolitik. Je nach Zensor werden nur bestimmte Politiken akzeptiert.
	 */
	public enum PolicyType {
		/**
		 * Potential Secrets und Secrecies sind gemischt.
		 */
		MIXED,
		/**
		 * Nur potentielle Geheimnisse.
		 */
		POTENTIAL_SECRETS,
		/**
		 * Nur Geheimnisse.
		 */
		SECRECIES,
		/**
		 * Noch nicht implementiert.
		 */
		EPISTEMIC;
		
		public String toString() {
			return super.toString().toLowerCase();
		}
	}
	
	/**
	 * Zeitraum, den eine Sicherheitspolitik bei Updates abdecken soll.
	 */
	public enum PolicyPreservation {
		/**
		 * Die Sicherheitspolitik garantiert, dass auch keine Inferenzen ueber einen
		 * vergangenen Zeitpunkt (vor dem Update) gemacht werden koennen.
		 */
		CONTINUOUS,
		/**
		 * Nur die aktuelle Datenbankinstanz (nach dem Update) wird geschuetzt.
		 * Beispiel: Kreditkartennummer. Sobald die alte Nummer auslaeuft, gesperrt
		 * und in der Datenbank geaendert wird, kann die alte Nummer bekannt werden.
		 */
		TEMPORARY;
		
		public String toString() {
			return super.toString().toLowerCase();
		}
	}
}
